package eu.europa.ec.taxud.cesop.validation;

import java.util.ArrayList;
import java.util.Collection;

import eu.europa.ec.taxud.cesop.domain.ValidationError;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorType;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeHolder;

import static eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum.CM_TR_0080;

/**
 * The type Cesop validation error list.
 * This class represents a list of ValidationError with a fixed maximum capacity.
 * Once the maximum capacity is reached, an error with code 50080 is added to the list (Too many errors).
 * After that no error can be added to list.
 *
 * @author SOFTDEV
 */
@SuppressWarnings("squid:S2160")
class CesopValidationErrorList extends ArrayList<ValidationError> {
    private static final long serialVersionUID = -6195714947208497354L;

    private boolean exceeded;
    private final int maxCapacity;

    /**
     * Instantiates a new Cesop validation error list.
     *
     * @param maxCapacity the max capacity
     */
    public CesopValidationErrorList(final int maxCapacity) {
        super();
        if (maxCapacity <= 0) {
            throw new IllegalArgumentException("Cannot create a list with negative or null capacity");
        }
        this.maxCapacity = maxCapacity;
    }

    @Override
    public boolean add(final ValidationError validationError) {
        if (this.exceeded) {
            throw new CesopTooManyErrorsException("Validation error list reached maximum capacity");
        }
        super.add(validationError);
        if (this.size() >= this.maxCapacity) {
            super.add(this.onExceed(validationError));
            this.exceeded = true;
            throw new CesopTooManyErrorsException("Validation error list reached maximum capacity");
        }
        return true;
    }

    @Override
    public boolean addAll(final Collection<? extends ValidationError> validationErrors) {
        for (final ValidationError validationError : validationErrors) {
            this.add(validationError);
        }
        return true;
    }

    private ValidationError onExceed(final ValidationError exceedingError) {
        final ValidationErrorType validationErrorType = ValidationErrorTypeHolder.INSTANCE.findByCode(CM_TR_0080.getCode());
        final ValidationError validationError = new ValidationError(CM_TR_0080);
        validationError.setErrorShortDesc(validationErrorType.getDescription());
        validationError.setErrorLongDesc(validationErrorType.getLongDescription());
        validationError.setMessageRefId(exceedingError.getMessageRefId());
        validationError.setErrorCounter(1);
        return validationError;
    }
}
